<?php

declare(strict_types=1);

namespace Erlage\Photogram\Requests\Post\Save;

use Exception;
use Erlage\Photogram\Settings;
use Erlage\Photogram\Data\Query;
use Erlage\Photogram\SystemLogger;
use Erlage\Photogram\Tools\Fso\Storage;
use Erlage\Photogram\Tools\Fso\ImageUploader;
use Erlage\Photogram\Constants\ServerConstants;
use Erlage\Photogram\Data\Models\Post\PostModel;
use Erlage\Photogram\Data\Models\User\UserModel;
use Erlage\Photogram\Constants\ResponseConstants;
use Erlage\Photogram\Exceptions\RequestException;
use Erlage\Photogram\Pattern\ExceptionalRequests;
use Erlage\Photogram\Data\Tables\Sys\RequestTable;
use Erlage\Photogram\Data\Tables\Post\PostSaveTable;
use Erlage\Photogram\Data\Dtos\Common\DisplayItemDTO;
use Erlage\Photogram\Data\Models\User\UserModelHelper;
use Erlage\Photogram\Data\Models\Post\Save\PostSaveModel;
use Erlage\Photogram\Data\Models\Post\Save\PostSaveBuilder;
use Erlage\Photogram\Data\Tables\Collection\CollectionTable;
use Erlage\Photogram\Data\Models\Collection\CollectionBuilder;
use Erlage\Photogram\Data\Dtos\Collection\CollectionDisplayImageDTO;

final class PostSaveActions extends ExceptionalRequests
{
    /*
    |--------------------------------------------------------------------------
    | do like
    |--------------------------------------------------------------------------
    */

    public static function add(): void
    {
        self::postSaveProcessor(true);
    }

    /*
    |--------------------------------------------------------------------------
    | undo like
    |--------------------------------------------------------------------------
    */

    public static function remove(): void
    {
        self::postSaveProcessor(false);
    }

    /*
    |--------------------------------------------------------------------------
    | processor
    |--------------------------------------------------------------------------
    */

    private static function postSaveProcessor(bool $doSave): void
    {
        self::process(function () use ($doSave)
        {
            /*
            |--------------------------------------------------------------------------
            | get data from request
            |--------------------------------------------------------------------------
            */

            $postIdFromReq = self::$request -> findKey(
                PostSaveTable::SAVED_POST_ID,
                RequestTable::PAYLOAD,
                PostSaveTable::TABLE_NAME
            );

            $collectionIdFromReq = self::$request -> findKey(
                PostSaveTable::SAVED_TO_COLLECTION_ID,
                RequestTable::PAYLOAD,
                PostSaveTable::TABLE_NAME
            );

            $collectionDisplayTitleFromReq = self::$request -> findKey(
                CollectionTable::DISPLAY_TITLE,
                RequestTable::PAYLOAD,
                CollectionTable::TABLE_NAME
            );

            self::ensureValue(ResponseConstants::ERROR_BAD_REQUEST_MSG, $postIdFromReq);

            $collectionIdProvided = self::isAvailable($collectionIdFromReq);

            $collectionDisplayTitleIsProvided = self::isAvailable($collectionDisplayTitleFromReq);

            /*
            |--------------------------------------------------------------------------
            | make sure user is authenticated
            |--------------------------------------------------------------------------
            */

            self::userEnsureAuthenticated();

            /*
            |--------------------------------------------------------------------------
            | if un-save request, simply remove the post from collection and return
            |--------------------------------------------------------------------------
            */

            if ( ! $doSave)
            {
                $postSaveTableDeleteQuery = (new Query())
                    -> from(PostSaveTable::TABLE_NAME)
                    -> where(PostSaveTable::SAVED_BY_USER_ID, self::$authedUserModel -> getId())
                    -> where(PostSaveTable::SAVED_POST_ID, $postIdFromReq);

                /*
                |--------------------------------------------------------------------------
                | limit delete to a single collection if provided
                | else it's going to delete post from all collections
                |--------------------------------------------------------------------------
                */

                if ($collectionIdProvided)
                {
                    $postSaveTableDeleteQuery -> where(PostSaveTable::SAVED_TO_COLLECTION_ID, $collectionIdFromReq);
                }

                /*
                |--------------------------------------------------------------------------
                | delete and return
                |--------------------------------------------------------------------------
                */

                $postSaveTableDeleteQuery -> delete();

                return;
            }

            /*
            |--------------------------------------------------------------------------
            | else it's a save request
            |--------------------------------------------------------------------------
            */

            /*
            |--------------------------------------------------------------------------
            | ensure post exists
            |--------------------------------------------------------------------------
            */

            $targetPostModel = PostModel::findFromId_throwException($postIdFromReq);

            /*
            |--------------------------------------------------------------------------
            | ensure post owner exists
            |--------------------------------------------------------------------------
            */

            $postOwnerUserModel = UserModel::findFromId_throwException($targetPostModel -> getOwnerUserId());

            /*
            |--------------------------------------------------------------------------
            | privacy checks
            |--------------------------------------------------------------------------
            */

            if ( ! UserModelHelper::isUserContentAvailable($postOwnerUserModel, self::$authedUserModel))
            {
                throw new RequestException(ResponseConstants::ERROR_BAD_REQUEST_MSG);
            }

            /*
            |--------------------------------------------------------------------------
            | if adding to a new collection
            |--------------------------------------------------------------------------
            */

            if ($collectionDisplayTitleIsProvided)
            {
                /*
                |--------------------------------------------------------------------------
                | build a collection
                |--------------------------------------------------------------------------
                */

                $newCollectionModel = (new CollectionBuilder())
                    -> setOwnerUserId(self::$authedUserModel -> getId())
                    -> setDisplayTitle($collectionDisplayTitleFromReq)
                    -> dispense();

                /*
                |--------------------------------------------------------------------------
                | user post image as collection cover
                |--------------------------------------------------------------------------
                */

                $postDisplayImage = $targetPostModel
                    -> getDisplayContent()
                    -> getFirstItem()
                    -> getDisplayItemDTO();

                /*
                |--------------------------------------------------------------------------
                | get raw contents
                |--------------------------------------------------------------------------
                */

                $postImageRawContents = Storage::disk($postDisplayImage -> getHost())
                    -> setObjectId($postDisplayImage -> getOriginalObjectId())
                    -> read();

                /*
                |--------------------------------------------------------------------------
                | create image uploader
                |--------------------------------------------------------------------------
                */

                $host = Settings::getString(ServerConstants::SS_ENUM_STORAGE_DISK_COLLECTION_IMAGES);
                $resolution = Settings::getInt(ServerConstants::SS_INT_COMPRESS_COLLECTION_COVER_IMAGE_FILE_RES);
                $compressedQuality = Settings::getInt(ServerConstants::SS_INT_COMPRESS_COLLECTION_COVER_IMAGE_FILE_QUALITY);

                $imageUploader = (new ImageUploader())
                    -> setHost($host)
                    -> setUserId(self::$authedUserModel -> getId())
                    -> setFilespace(DisplayItemDTO::FILESPACE_COLLECTION)
                    -> setSaveResolution($resolution)
                    -> setCompressedQuality($compressedQuality)
                    -> setRawContents($postImageRawContents)
                    -> setExtensionFromIdentifier($postDisplayImage -> getIdentifier());

                /*
                |--------------------------------------------------------------------------
                | finish save
                |--------------------------------------------------------------------------
                */

                $uploadHandle = $imageUploader -> process();

                // check if image upload has succeeded

                if ( ! $uploadHandle -> processed)
                {
                    // upload failed, log internally

                    SystemLogger::internalException(new Exception($uploadHandle -> error));

                    // something went wrong

                    throw new RequestException(ResponseConstants::ERROR_BAD_REQUEST_MSG);
                }

                // else get DTO object

                $uploadedImageDTO = $imageUploader -> getDisplayItemDTO();

                /*
                |--------------------------------------------------------------------------
                | update model
                |--------------------------------------------------------------------------
                */

                $newCollectionModel -> update(
                    array(
                        CollectionTable::DISPLAY_IMAGE => (new CollectionDisplayImageDTO())
                            -> setHost($uploadedImageDTO -> getHost())
                            -> setIdentifier($uploadedImageDTO -> getIdentifier()),
                    )
                );

                /*
                |--------------------------------------------------------------------------
                | save collection
                |--------------------------------------------------------------------------
                */

                $newCollectionModel -> save();

                /*
                |--------------------------------------------------------------------------
                | build post save
                |--------------------------------------------------------------------------
                */

                $postSaveModel = (new PostSaveBuilder())
                    -> setSavedPostId($targetPostModel -> getId())
                    -> setSavedByUserId(self::$authedUserModel -> getId())
                    -> setSavedToCollectionId($newCollectionModel -> getId())
                    -> dispense();

                /*
                |--------------------------------------------------------------------------
                | save
                |--------------------------------------------------------------------------
                */

                $postSaveModel -> save();

                /*
                |--------------------------------------------------------------------------
                |  if everything is alright, add to response and return
                |--------------------------------------------------------------------------
                */

                self::addToResponse(PostSaveTable::getTableName(), $postSaveModel -> getDataMap());

                return;
            }

            /*
            |--------------------------------------------------------------------------
            | else save it to collection id provided or all
            |--------------------------------------------------------------------------
            */

            /*
            |--------------------------------------------------------------------------
            | make sure post is not already saved
            |--------------------------------------------------------------------------
            */

            $postSaveBeanQuery = (new Query())
                -> from(PostSaveTable::TABLE_NAME)
                -> where(PostSaveTable::SAVED_POST_ID, $targetPostModel -> getId())
                -> where(PostSaveTable::SAVED_BY_USER_ID, self::$authedUserModel -> getId());

            if ($collectionIdProvided)
            {
                $postSaveBeanQuery -> where(PostSaveTable::SAVED_TO_COLLECTION_ID, $collectionIdFromReq);
            }

            $postSaveBean = $postSaveBeanQuery -> selectOne();

            /*
            |--------------------------------------------------------------------------
            | if post is already saved
            |--------------------------------------------------------------------------
            */

            if (null != $postSaveBean)
            {
                /*
                |--------------------------------------------------------------------------
                | create post save model from selected bean
                |--------------------------------------------------------------------------
                */

                $postSaveModel = PostSaveModel::createFromUntouchedBean_noException($postSaveBean);

                self::ensureModel($postSaveModel);

                /*
                |--------------------------------------------------------------------------
                | add to post save model to response
                |--------------------------------------------------------------------------
                */

                self::addToResponse(PostSaveTable::getTableName(), $postSaveModel -> getDataMap());

                return;
            }

            /*
            |--------------------------------------------------------------------------
            | else if post is not already saved
            |--------------------------------------------------------------------------
            */

            /*
            |--------------------------------------------------------------------------
            | build post save
            |--------------------------------------------------------------------------
            */

            $postSaveModelBuilder = (new PostSaveBuilder())
                -> setSavedPostId($targetPostModel -> getId())
                -> setSavedByUserId(self::$authedUserModel -> getId());

            if ($collectionIdProvided)
            {
                $postSaveModelBuilder -> setSavedToCollectionId($collectionIdFromReq);
            }

            $postSaveModel = $postSaveModelBuilder -> dispense();

            /*
            |--------------------------------------------------------------------------
            | save
            |--------------------------------------------------------------------------
            */

            $postSaveModel -> save();

            /*
            |--------------------------------------------------------------------------
            |  if everything is alright, add to response
            |--------------------------------------------------------------------------
            */

            self::addToResponse(PostSaveTable::getTableName(), $postSaveModel -> getDataMap());
        });
    }
}
